import os
import json
import argparse

import numpy as np
from tqdm import tqdm

from preprocess import preprocess, postprocess


class WrapRunner(object):

    def __init__(self, runner, rbo_mode, id=None):
        self.runner = runner
        self.rbo_mode = rbo_mode
        self.id = id
    
    def __call__(self, *args):
        if self.rbo_mode:
            inputs = list(args)
            for i in range(len(inputs)):
                if not isinstance(inputs[i], np.ndarray):
                    inputs[i] = inputs[i].numpy()
            outs = self.runner.inference(self.id, inputs)
            if isinstance(outs, list) and len(outs) == 1:
                return outs[0]
            return outs
        else:
            outs = self.runner.run(*args, as_numpy=True)
            return outs

    def close(self):
        if self.rbo_mode:
            self.runner.remove_model(self.id)


def init_runner(args):
    suffix = os.path.splitext(args.model_file)[1]
    print(f"INFO: Load model from {args.model_file}")
    if suffix == '.sg':
        import rbcc
        from rbcc.backend.runner import SGRunner
        sg = rbcc.load_sg(args.model_file)
        device = f"cuda:{args.device_id}" if args.device_id >= 0 else "cpu"
        runner = SGRunner(sg, device=device, gc_collect=True)
        return WrapRunner(runner, False)
    elif suffix == '.rbo':
        from rboexec.core.client import GRPCClient
        client = GRPCClient(os.environ['CAISA_REMOTE_IP'], int(os.environ['CAISA_REMOTE_PORT']))
        id = client.upload_model(model_path=os.path.abspath(args.model_file), device_id=args.device_id)
        return WrapRunner(client, True, id)
    else:
        raise ValueError(f"excepted .sg or .rbo file, not {suffix}")

def generate_test_dataset(args, transform):
    root = args.data
    data_dir = os.path.join(root, 'test_npy')
    label_path = os.path.join(root, "classInd.json")
    with open(label_path) as f:  labels = json.load(f)
    for video_name in sorted(os.listdir(data_dir)):
        label_name = video_name.split("_")[1].strip()
        label = labels.get(label_name)
        if not label:  continue
        video_path = os.path.join(data_dir, video_name)
        frames = transform(video_path)
        yield frames, label

def main(args):
    print("INFO: Init Runner.")
    runner = init_runner(args)

    print("INFO: Load dataset.")
    val_loader = generate_test_dataset(args, preprocess)
    data_dir = os.path.join(args.data, 'test_npy')
    num_samples = len(os.listdir(data_dir))

    # 开始进行评价
    print("INFO: Start inference.")
    pbar = tqdm(desc=f"Eval in UCF101", unit='img', total=num_samples)
    total, correct = 0, 0
    for index, (frames, label) in enumerate(val_loader):
        outputs = runner(frames)
        outputs = postprocess(outputs)
        if outputs == label:
            correct += 1
        total += 1
        pbar.update(1)

    pbar.close()
    runner.close()
    acc = correct / total
    acc = round(acc * 100, 2)
    print(f"INFO: Acc: {acc}", flush=True)


if __name__ == '__main__':
    parser = argparse.ArgumentParser()
    parser.add_argument('model_file', type=str, help='rbo or sg file path.')
    parser.add_argument('--data', type=str, default=None, help='dataset dir.')
    parser.add_argument('--device-id', type=int, default=0, help="device id.")
    main(parser.parse_args())
